using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;
using GlampingParadise.Data;
using GlampingParadise.Models;

namespace GlampingParadise.Controllers
{
    [Authorize(Roles = "Admin")]
    public class AdminController : Controller
    {
        private readonly ApplicationDbContext _context;

        public AdminController(ApplicationDbContext context)
        {
            _context = context;
        }

        public async Task<IActionResult> Index()
        {
            var stats = new
            {
                TotalReservations = await _context.Reservations.CountAsync(),
                PendingReservations = await _context.Reservations.CountAsync(r => r.Status == ReservationStatus.Pending),
                TotalCabins = await _context.Cabins.CountAsync(),
                AvailableCabins = await _context.Cabins.CountAsync(c => c.IsAvailable),
                TotalPlans = await _context.Plans.CountAsync(),
                ActivePlans = await _context.Plans.CountAsync(p => p.IsActive),
                TotalUsers = await _context.Users.CountAsync()
            };

            ViewBag.Stats = stats;

            var recentReservations = await _context.Reservations
                .Include(r => r.User)
                .Include(r => r.Cabin)
                .Include(r => r.Plan)
                .OrderByDescending(r => r.ReservationDate)
                .Take(5)
                .ToListAsync();

            return View(recentReservations);
        }

        #region Gestión de Cabañas

        public async Task<IActionResult> Cabins()
        {
            var cabins = await _context.Cabins.OrderBy(c => c.Name).ToListAsync();
            return View(cabins);
        }

        public IActionResult CreateCabin()
        {
            return View();
        }

        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> CreateCabin(Cabin cabin)
        {
            if (ModelState.IsValid)
            {
                cabin.CreatedDate = DateTime.Now;
                _context.Add(cabin);
                await _context.SaveChangesAsync();
                TempData["Success"] = "Cabaña creada exitosamente";
                return RedirectToAction(nameof(Cabins));
            }
            return View(cabin);
        }

        public async Task<IActionResult> EditCabin(int? id)
        {
            if (id == null)
            {
                return NotFound();
            }

            var cabin = await _context.Cabins.FindAsync(id);
            if (cabin == null)
            {
                return NotFound();
            }
            return View(cabin);
        }

        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> EditCabin(int id, Cabin cabin)
        {
            if (id != cabin.Id)
            {
                return NotFound();
            }

            if (ModelState.IsValid)
            {
                try
                {
                    var existingCabin = await _context.Cabins.FindAsync(id);
                    if (existingCabin == null)
                    {
                        return NotFound();
                    }

                    existingCabin.Name = cabin.Name;
                    existingCabin.Description = cabin.Description;
                    existingCabin.Capacity = cabin.Capacity;
                    existingCabin.PricePerNight = cabin.PricePerNight;
                    existingCabin.ImageUrl = cabin.ImageUrl;
                    existingCabin.IsAvailable = cabin.IsAvailable;
                    existingCabin.Amenities = cabin.Amenities;

                    _context.Update(existingCabin);
                    await _context.SaveChangesAsync();
                    TempData["Success"] = "Cabaña actualizada exitosamente";
                    return RedirectToAction(nameof(Cabins));
                }
                catch (DbUpdateConcurrencyException)
                {
                    if (!CabinExists(cabin.Id))
                    {
                        return NotFound();
                    }
                    else
                    {
                        throw;
                    }
                }
            }
            return View(cabin);
        }

        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> DeleteCabin(int id)
        {
            var cabin = await _context.Cabins.FindAsync(id);
            if (cabin != null)
            {
                // Verificar si tiene reservas activas
                var activeReservations = await _context.Reservations
                    .Where(r => r.CabinId == id && r.Status != ReservationStatus.Cancelled && r.Status != ReservationStatus.Completed)
                    .CountAsync();

                if (activeReservations > 0)
                {
                    TempData["Error"] = "No se puede eliminar la cabaña porque tiene reservas activas";
                    return RedirectToAction(nameof(Cabins));
                }

                _context.Cabins.Remove(cabin);
                await _context.SaveChangesAsync();
                TempData["Success"] = "Cabaña eliminada exitosamente";
            }

            return RedirectToAction(nameof(Cabins));
        }

        #endregion

        #region Gestión de Planes

        public async Task<IActionResult> Plans()
        {
            var plans = await _context.Plans.OrderBy(p => p.Name).ToListAsync();
            return View(plans);
        }

        public IActionResult CreatePlan()
        {
            return View();
        }

        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> CreatePlan(Plan plan)
        {
            if (ModelState.IsValid)
            {
                plan.CreatedDate = DateTime.Now;
                _context.Add(plan);
                await _context.SaveChangesAsync();
                TempData["Success"] = "Plan creado exitosamente";
                return RedirectToAction(nameof(Plans));
            }
            return View(plan);
        }

        public async Task<IActionResult> EditPlan(int? id)
        {
            if (id == null)
            {
                return NotFound();
            }

            var plan = await _context.Plans.FindAsync(id);
            if (plan == null)
            {
                return NotFound();
            }
            return View(plan);
        }

        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> EditPlan(int id, Plan plan)
        {
            if (id != plan.Id)
            {
                return NotFound();
            }

            if (ModelState.IsValid)
            {
                try
                {
                    var existingPlan = await _context.Plans.FindAsync(id);
                    if (existingPlan == null)
                    {
                        return NotFound();
                    }

                    existingPlan.Name = plan.Name;
                    existingPlan.Description = plan.Description;
                    existingPlan.Price = plan.Price;
                    existingPlan.Duration = plan.Duration;
                    existingPlan.ImageUrl = plan.ImageUrl;
                    existingPlan.IsActive = plan.IsActive;
                    existingPlan.IncludedActivities = plan.IncludedActivities;

                    _context.Update(existingPlan);
                    await _context.SaveChangesAsync();
                    TempData["Success"] = "Plan actualizado exitosamente";
                    return RedirectToAction(nameof(Plans));
                }
                catch (DbUpdateConcurrencyException)
                {
                    if (!PlanExists(plan.Id))
                    {
                        return NotFound();
                    }
                    else
                    {
                        throw;
                    }
                }
            }
            return View(plan);
        }

        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> DeletePlan(int id)
        {
            var plan = await _context.Plans.FindAsync(id);
            if (plan != null)
            {
                // Verificar si tiene reservas activas
                var activeReservations = await _context.Reservations
                    .Where(r => r.PlanId == id && r.Status != ReservationStatus.Cancelled && r.Status != ReservationStatus.Completed)
                    .CountAsync();

                if (activeReservations > 0)
                {
                    TempData["Error"] = "No se puede eliminar el plan porque tiene reservas activas";
                    return RedirectToAction(nameof(Plans));
                }

                _context.Plans.Remove(plan);
                await _context.SaveChangesAsync();
                TempData["Success"] = "Plan eliminado exitosamente";
            }

            return RedirectToAction(nameof(Plans));
        }

        #endregion

        #region Gestión de Reservas

        public async Task<IActionResult> Reservations()
        {
            var reservations = await _context.Reservations
                .Include(r => r.User)
                .Include(r => r.Cabin)
                .Include(r => r.Plan)
                .OrderByDescending(r => r.ReservationDate)
                .ToListAsync();

            return View(reservations);
        }

        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> UpdateReservationStatus(int id, ReservationStatus status)
        {
            var reservation = await _context.Reservations.FindAsync(id);
            if (reservation == null)
            {
                return NotFound();
            }

            reservation.Status = status;
            reservation.LastModified = DateTime.Now;

            _context.Update(reservation);
            await _context.SaveChangesAsync();

            TempData["Success"] = "Estado de la reserva actualizado exitosamente";
            return RedirectToAction(nameof(Reservations));
        }

        #endregion

        private bool CabinExists(int id)
        {
            return _context.Cabins.Any(e => e.Id == id);
        }

        private bool PlanExists(int id)
        {
            return _context.Plans.Any(e => e.Id == id);
        }
    }
}