using Microsoft.AspNetCore.Identity.EntityFrameworkCore;
using Microsoft.EntityFrameworkCore;
using GlampingParadise.Models;

namespace GlampingParadise.Data
{
    public class ApplicationDbContext : IdentityDbContext<ApplicationUser>
    {
        public ApplicationDbContext(DbContextOptions<ApplicationDbContext> options)
            : base(options)
        {
        }

        public DbSet<Cabin> Cabins { get; set; }
        public DbSet<Plan> Plans { get; set; }
        public DbSet<Reservation> Reservations { get; set; }
        public DbSet<Reserva> Reservas { get; set; }

        protected override void OnModelCreating(ModelBuilder builder)
        {
            base.OnModelCreating(builder);

            // Configuración de relaciones
            builder.Entity<Reservation>()
                .HasOne(r => r.User)
                .WithMany(u => u.Reservations)
                .HasForeignKey(r => r.UserId)
                .OnDelete(DeleteBehavior.Restrict);

            builder.Entity<Reservation>()
                .HasOne(r => r.Cabin)
                .WithMany(c => c.Reservations)
                .HasForeignKey(r => r.CabinId)
                .OnDelete(DeleteBehavior.Restrict);

            builder.Entity<Reservation>()
                .HasOne(r => r.Plan)
                .WithMany(p => p.Reservations)
                .HasForeignKey(r => r.PlanId)
                .OnDelete(DeleteBehavior.SetNull);

            // Configuración de precisión decimal
            builder.Entity<Cabin>()
                .Property(c => c.PricePerNight)
                .HasPrecision(18, 2);

            builder.Entity<Plan>()
                .Property(p => p.Price)
                .HasPrecision(18, 2);

            builder.Entity<Reservation>()
                .Property(r => r.TotalPrice)
                .HasPrecision(18, 2);

            // Datos semilla para cabañas
            builder.Entity<Cabin>().HasData(
                new Cabin
                {
                    Id = 1,
                    Name = "Cabaña Romántica",
                    Description = "Perfecta para parejas, con vista al lago y jacuzzi privado",
                    Capacity = 2,
                    PricePerNight = 150.00m,
                    ImageUrl = "/images/cabin1.jpg",
                    IsAvailable = true,
                    Amenities = "Jacuzzi, Vista al lago, Chimenea, WiFi, Aire acondicionado",
                    CreatedDate = DateTime.Now
                },
                new Cabin
                {
                    Id = 2,
                    Name = "Cabaña Familiar",
                    Description = "Espaciosa cabaña para familias con área de juegos",
                    Capacity = 6,
                    PricePerNight = 250.00m,
                    ImageUrl = "/images/cabin2.jpg",
                    IsAvailable = true,
                    Amenities = "Área de juegos, Cocina completa, 2 habitaciones, WiFi, Parrilla",
                    CreatedDate = DateTime.Now
                },
                new Cabin
                {
                    Id = 3,
                    Name = "Cabaña Aventura",
                    Description = "Para los amantes de la naturaleza y deportes extremos",
                    Capacity = 4,
                    PricePerNight = 200.00m,
                    ImageUrl = "/images/cabin3.jpg",
                    IsAvailable = true,
                    Amenities = "Equipos de aventura, Senderos privados, Fogata, WiFi",
                    CreatedDate = DateTime.Now
                }
            );

            // Datos semilla para planes
            builder.Entity<Plan>().HasData(
                new Plan
                {
                    Id = 1,
                    Name = "Plan Romántico",
                    Description = "Incluye cena romántica, spa de parejas y tour en bote",
                    Price = 300.00m,
                    Duration = 2,
                    ImageUrl = "/images/plan1.jpg",
                    IsActive = true,
                    IncludedActivities = "Cena romántica, Spa de parejas, Tour en bote, Desayuno en la cabaña",
                    CreatedDate = DateTime.Now
                },
                new Plan
                {
                    Id = 2,
                    Name = "Plan Familiar",
                    Description = "Actividades para toda la familia con entretenimiento para niños",
                    Price = 450.00m,
                    Duration = 3,
                    ImageUrl = "/images/plan2.jpg",
                    IsActive = true,
                    IncludedActivities = "Actividades para niños, Senderismo familiar, Fogata nocturna, Todas las comidas",
                    CreatedDate = DateTime.Now
                },
                new Plan
                {
                    Id = 3,
                    Name = "Plan Aventura",
                    Description = "Para los amantes de la adrenalina y deportes extremos",
                    Price = 500.00m,
                    Duration = 3,
                    ImageUrl = "/images/plan3.jpg",
                    IsActive = true,
                    IncludedActivities = "Canopy, Rappel, Kayak, Senderismo extremo, Camping nocturno",
                    CreatedDate = DateTime.Now
                }
            );
        }
    }
}