using System.ComponentModel.DataAnnotations;
using System.ComponentModel.DataAnnotations.Schema;

namespace GlampingParadise.Models
{
    public enum ReservationStatus
    {
        [Display(Name = "Pendiente")]
        Pending,
        [Display(Name = "Confirmada")]
        Confirmed,
        [Display(Name = "Cancelada")]
        Cancelled,
        [Display(Name = "Completada")]
        Completed
    }

    public class Reservation
    {
        [Key]
        public int Id { get; set; }

        [Required(ErrorMessage = "El usuario es obligatorio")]
        [Display(Name = "Usuario")]
        public string UserId { get; set; } = string.Empty;

        [Required(ErrorMessage = "La cabaña es obligatoria")]
        [Display(Name = "Cabaña")]
        public int CabinId { get; set; }

        [Display(Name = "Plan")]
        public int? PlanId { get; set; }

        [Required(ErrorMessage = "La fecha de check-in es obligatoria")]
        [Display(Name = "Fecha de check-in")]
        [DataType(DataType.Date)]
        public DateTime CheckInDate { get; set; }

        [Required(ErrorMessage = "La fecha de check-out es obligatoria")]
        [Display(Name = "Fecha de check-out")]
        [DataType(DataType.Date)]
        public DateTime CheckOutDate { get; set; }

        [Required(ErrorMessage = "El número de huéspedes es obligatorio")]
        [Range(1, 20, ErrorMessage = "El número de huéspedes debe estar entre 1 y 20")]
        [Display(Name = "Número de huéspedes")]
        public int NumberOfGuests { get; set; }

        [Required(ErrorMessage = "El precio total es obligatorio")]
        [Range(0.01, 100000, ErrorMessage = "El precio total debe ser mayor a 0")]
        [Column(TypeName = "decimal(18,2)")]
        [Display(Name = "Precio total")]
        public decimal TotalPrice { get; set; }

        [Required(ErrorMessage = "El estado es obligatorio")]
        [Display(Name = "Estado")]
        public ReservationStatus Status { get; set; } = ReservationStatus.Pending;

        [StringLength(500, ErrorMessage = "Las observaciones no pueden exceder 500 caracteres")]
        [Display(Name = "Observaciones")]
        public string? Notes { get; set; }

        [Display(Name = "Fecha de reserva")]
        public DateTime ReservationDate { get; set; } = DateTime.Now;

        [Display(Name = "Última modificación")]
        public DateTime LastModified { get; set; } = DateTime.Now;

        // Navegación
        [ForeignKey("UserId")]
        public virtual ApplicationUser User { get; set; } = null!;

        [ForeignKey("CabinId")]
        public virtual Cabin Cabin { get; set; } = null!;

        [ForeignKey("PlanId")]
        public virtual Plan? Plan { get; set; }

        // Validación personalizada
        public bool IsValidDateRange()
        {
            return CheckOutDate > CheckInDate && CheckInDate >= DateTime.Today;
        }

        public int GetNumberOfNights()
        {
            return (CheckOutDate - CheckInDate).Days;
        }
    }
}