using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Identity;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;
using GlampingParadise.Data;
using GlampingParadise.Models;

namespace GlampingParadise.Controllers
{
    [Authorize]
    public class ReservationsController : Controller
    {
        private readonly ApplicationDbContext _context;
        private readonly UserManager<ApplicationUser> _userManager;

        public ReservationsController(ApplicationDbContext context, UserManager<ApplicationUser> userManager)
        {
            _context = context;
            _userManager = userManager;
        }

        // GET: Reservations
        public async Task<IActionResult> Index()
        {
            var user = await _userManager.GetUserAsync(User);
            if (user == null)
            {
                return RedirectToAction("Login", "Account");
            }

            var reservations = await _context.Reservations
                .Include(r => r.Cabin)
                .Include(r => r.Plan)
                .Where(r => r.UserId == user.Id)
                .OrderByDescending(r => r.ReservationDate)
                .ToListAsync();

            return View(reservations);
        }

        // GET: Reservations/Create
        public async Task<IActionResult> Create(int? cabinId, int? planId)
        {
            var reservation = new Reservation();
            
            if (cabinId.HasValue)
            {
                var cabin = await _context.Cabins.FindAsync(cabinId.Value);
                if (cabin != null && cabin.IsAvailable)
                {
                    reservation.CabinId = cabinId.Value;
                    ViewBag.SelectedCabin = cabin;
                }
            }

            if (planId.HasValue)
            {
                var plan = await _context.Plans.FindAsync(planId.Value);
                if (plan != null && plan.IsActive)
                {
                    reservation.PlanId = planId.Value;
                    ViewBag.SelectedPlan = plan;
                }
            }

            ViewBag.Cabins = await _context.Cabins.Where(c => c.IsAvailable).ToListAsync();
            ViewBag.Plans = await _context.Plans.Where(p => p.IsActive).ToListAsync();

            return View(reservation);
        }

        // POST: Reservations/Create
        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> Create(Reservation reservation)
        {
            var user = await _userManager.GetUserAsync(User);
            if (user == null)
            {
                return RedirectToAction("Login", "Account");
            }

            reservation.UserId = user.Id;
            reservation.ReservationDate = DateTime.Now;
            reservation.LastModified = DateTime.Now;
            reservation.Status = ReservationStatus.Pending;

            // Validar fechas
            if (reservation.CheckOutDate <= reservation.CheckInDate)
            {
                ModelState.AddModelError("CheckOutDate", "La fecha de check-out debe ser posterior a la fecha de check-in");
            }

            if (reservation.CheckInDate < DateTime.Today)
            {
                ModelState.AddModelError("CheckInDate", "La fecha de check-in no puede ser anterior a hoy");
            }

            // Verificar disponibilidad de la cabaña
            var cabin = await _context.Cabins.FindAsync(reservation.CabinId);
            if (cabin == null || !cabin.IsAvailable)
            {
                ModelState.AddModelError("CabinId", "La cabaña seleccionada no está disponible");
            }

            // Verificar capacidad
            if (cabin != null && reservation.NumberOfGuests > cabin.Capacity)
            {
                ModelState.AddModelError("NumberOfGuests", $"El número de huéspedes no puede exceder la capacidad de la cabaña ({cabin.Capacity})");
            }

            // Verificar conflictos de fechas
            var conflictingReservations = await _context.Reservations
                .Where(r => r.CabinId == reservation.CabinId && 
                           r.Status != ReservationStatus.Cancelled &&
                           ((r.CheckInDate <= reservation.CheckInDate && r.CheckOutDate > reservation.CheckInDate) ||
                            (r.CheckInDate < reservation.CheckOutDate && r.CheckOutDate >= reservation.CheckOutDate) ||
                            (r.CheckInDate >= reservation.CheckInDate && r.CheckOutDate <= reservation.CheckOutDate)))
                .ToListAsync();

            if (conflictingReservations.Any())
            {
                ModelState.AddModelError("", "La cabaña no está disponible en las fechas seleccionadas");
            }

            // Calcular precio total
            if (cabin != null)
            {
                var nights = (reservation.CheckOutDate - reservation.CheckInDate).Days;
                reservation.TotalPrice = cabin.PricePerNight * nights;

                if (reservation.PlanId.HasValue)
                {
                    var plan = await _context.Plans.FindAsync(reservation.PlanId.Value);
                    if (plan != null && plan.IsActive)
                    {
                        reservation.TotalPrice += plan.Price;
                    }
                }
            }

            if (ModelState.IsValid)
            {
                _context.Add(reservation);
                await _context.SaveChangesAsync();
                TempData["Success"] = "Reserva creada exitosamente";
                return RedirectToAction(nameof(Index));
            }

            ViewBag.Cabins = await _context.Cabins.Where(c => c.IsAvailable).ToListAsync();
            ViewBag.Plans = await _context.Plans.Where(p => p.IsActive).ToListAsync();
            return View(reservation);
        }

        // GET: Reservations/Edit/5
        public async Task<IActionResult> Edit(int? id)
        {
            if (id == null)
            {
                return NotFound();
            }

            var user = await _userManager.GetUserAsync(User);
            var reservation = await _context.Reservations
                .Include(r => r.Cabin)
                .Include(r => r.Plan)
                .FirstOrDefaultAsync(r => r.Id == id && r.UserId == user.Id);

            if (reservation == null)
            {
                return NotFound();
            }

            // Solo permitir editar reservas pendientes o confirmadas
            if (reservation.Status == ReservationStatus.Cancelled || reservation.Status == ReservationStatus.Completed)
            {
                TempData["Error"] = "No se puede editar una reserva cancelada o completada";
                return RedirectToAction(nameof(Index));
            }

            ViewBag.Cabins = await _context.Cabins.Where(c => c.IsAvailable).ToListAsync();
            ViewBag.Plans = await _context.Plans.Where(p => p.IsActive).ToListAsync();

            return View(reservation);
        }

        // POST: Reservations/Edit/5
        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> Edit(int id, Reservation reservation)
        {
            if (id != reservation.Id)
            {
                return NotFound();
            }

            var user = await _userManager.GetUserAsync(User);
            var existingReservation = await _context.Reservations
                .FirstOrDefaultAsync(r => r.Id == id && r.UserId == user.Id);

            if (existingReservation == null)
            {
                return NotFound();
            }

            // Solo permitir editar reservas pendientes o confirmadas
            if (existingReservation.Status == ReservationStatus.Cancelled || existingReservation.Status == ReservationStatus.Completed)
            {
                TempData["Error"] = "No se puede editar una reserva cancelada o completada";
                return RedirectToAction(nameof(Index));
            }

            // Validar fechas
            if (reservation.CheckOutDate <= reservation.CheckInDate)
            {
                ModelState.AddModelError("CheckOutDate", "La fecha de check-out debe ser posterior a la fecha de check-in");
            }

            if (reservation.CheckInDate < DateTime.Today)
            {
                ModelState.AddModelError("CheckInDate", "La fecha de check-in no puede ser anterior a hoy");
            }

            // Verificar disponibilidad de la cabaña
            var cabin = await _context.Cabins.FindAsync(reservation.CabinId);
            if (cabin == null || !cabin.IsAvailable)
            {
                ModelState.AddModelError("CabinId", "La cabaña seleccionada no está disponible");
            }

            // Verificar capacidad
            if (cabin != null && reservation.NumberOfGuests > cabin.Capacity)
            {
                ModelState.AddModelError("NumberOfGuests", $"El número de huéspedes no puede exceder la capacidad de la cabaña ({cabin.Capacity})");
            }

            // Verificar conflictos de fechas (excluyendo la reserva actual)
            var conflictingReservations = await _context.Reservations
                .Where(r => r.CabinId == reservation.CabinId && 
                           r.Id != id &&
                           r.Status != ReservationStatus.Cancelled &&
                           ((r.CheckInDate <= reservation.CheckInDate && r.CheckOutDate > reservation.CheckInDate) ||
                            (r.CheckInDate < reservation.CheckOutDate && r.CheckOutDate >= reservation.CheckOutDate) ||
                            (r.CheckInDate >= reservation.CheckInDate && r.CheckOutDate <= reservation.CheckOutDate)))
                .ToListAsync();

            if (conflictingReservations.Any())
            {
                ModelState.AddModelError("", "La cabaña no está disponible en las fechas seleccionadas");
            }

            // Calcular precio total
            if (cabin != null)
            {
                var nights = (reservation.CheckOutDate - reservation.CheckInDate).Days;
                reservation.TotalPrice = cabin.PricePerNight * nights;

                if (reservation.PlanId.HasValue)
                {
                    var plan = await _context.Plans.FindAsync(reservation.PlanId.Value);
                    if (plan != null && plan.IsActive)
                    {
                        reservation.TotalPrice += plan.Price;
                    }
                }
            }

            if (ModelState.IsValid)
            {
                try
                {
                    existingReservation.CabinId = reservation.CabinId;
                    existingReservation.PlanId = reservation.PlanId;
                    existingReservation.CheckInDate = reservation.CheckInDate;
                    existingReservation.CheckOutDate = reservation.CheckOutDate;
                    existingReservation.NumberOfGuests = reservation.NumberOfGuests;
                    existingReservation.TotalPrice = reservation.TotalPrice;
                    existingReservation.Notes = reservation.Notes;
                    existingReservation.LastModified = DateTime.Now;

                    _context.Update(existingReservation);
                    await _context.SaveChangesAsync();
                    TempData["Success"] = "Reserva actualizada exitosamente";
                    return RedirectToAction(nameof(Index));
                }
                catch (DbUpdateConcurrencyException)
                {
                    if (!ReservationExists(reservation.Id))
                    {
                        return NotFound();
                    }
                    else
                    {
                        throw;
                    }
                }
            }

            ViewBag.Cabins = await _context.Cabins.Where(c => c.IsAvailable).ToListAsync();
            ViewBag.Plans = await _context.Plans.Where(p => p.IsActive).ToListAsync();
            return View(reservation);
        }

        // POST: Reservations/Cancel/5
        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> Cancel(int id)
        {
            var user = await _userManager.GetUserAsync(User);
            var reservation = await _context.Reservations
                .FirstOrDefaultAsync(r => r.Id == id && r.UserId == user.Id);

            if (reservation == null)
            {
                return NotFound();
            }

            if (reservation.Status == ReservationStatus.Cancelled)
            {
                TempData["Error"] = "La reserva ya está cancelada";
                return RedirectToAction(nameof(Index));
            }

            if (reservation.Status == ReservationStatus.Completed)
            {
                TempData["Error"] = "No se puede cancelar una reserva completada";
                return RedirectToAction(nameof(Index));
            }

            reservation.Status = ReservationStatus.Cancelled;
            reservation.LastModified = DateTime.Now;

            _context.Update(reservation);
            await _context.SaveChangesAsync();

            TempData["Success"] = "Reserva cancelada exitosamente";
            return RedirectToAction(nameof(Index));
        }

        private bool ReservationExists(int id)
        {
            return _context.Reservations.Any(e => e.Id == id);
        }
    }
}