// JavaScript personalizado para Glamping Paradise

// Función para mostrar/ocultar contraseñas
function togglePassword(fieldId) {
    const field = document.getElementById(fieldId);
    const icon = document.getElementById(fieldId + '-icon');
    
    if (field.type === 'password') {
        field.type = 'text';
        icon.classList.remove('fa-eye');
        icon.classList.add('fa-eye-slash');
    } else {
        field.type = 'password';
        icon.classList.remove('fa-eye-slash');
        icon.classList.add('fa-eye');
    }
}

// Función para calcular precios en tiempo real
function calculatePrice() {
    const checkIn = document.getElementById('CheckInDate')?.value;
    const checkOut = document.getElementById('CheckOutDate')?.value;
    const guests = document.getElementById('NumberOfGuests')?.value || 1;
    const cabinPrice = parseFloat(document.getElementById('cabin-price')?.dataset.price || 0);
    const planPrice = parseFloat(document.getElementById('plan-price')?.dataset.price || 0);

    if (checkIn && checkOut) {
        const startDate = new Date(checkIn);
        const endDate = new Date(checkOut);
        const timeDiff = endDate.getTime() - startDate.getTime();
        const daysDiff = Math.ceil(timeDiff / (1000 * 3600 * 24));

        if (daysDiff > 0) {
            const totalCabinPrice = cabinPrice * daysDiff;
            const totalPlanPrice = planPrice * guests;
            const totalPrice = totalCabinPrice + totalPlanPrice;

            // Actualizar elementos de precio en la página
            const priceElement = document.getElementById('total-price');
            if (priceElement) {
                priceElement.textContent = '$' + totalPrice.toLocaleString();
            }

            const detailsElement = document.getElementById('price-details');
            if (detailsElement) {
                detailsElement.innerHTML = `
                    <div class="price-breakdown">
                        <div class="d-flex justify-content-between">
                            <span>Cabaña (${daysDiff} noches):</span>
                            <span>$${totalCabinPrice.toLocaleString()}</span>
                        </div>
                        ${totalPlanPrice > 0 ? `
                        <div class="d-flex justify-content-between">
                            <span>Plan (${guests} personas):</span>
                            <span>$${totalPlanPrice.toLocaleString()}</span>
                        </div>
                        ` : ''}
                        <hr>
                        <div class="d-flex justify-content-between fw-bold">
                            <span>Total:</span>
                            <span>$${totalPrice.toLocaleString()}</span>
                        </div>
                    </div>
                `;
            }

            // Actualizar campo oculto para el formulario
            const hiddenPriceField = document.getElementById('TotalPrice');
            if (hiddenPriceField) {
                hiddenPriceField.value = totalPrice;
            }
        }
    }
}

// Función para filtrar elementos
function filterItems(containerSelector, filterFunction) {
    const container = document.querySelector(containerSelector);
    if (!container) return;

    const items = container.querySelectorAll('.filter-item');
    let visibleCount = 0;

    items.forEach(item => {
        if (filterFunction(item)) {
            item.style.display = 'block';
            item.classList.add('fade-in-up');
            visibleCount++;
        } else {
            item.style.display = 'none';
            item.classList.remove('fade-in-up');
        }
    });

    // Mostrar mensaje si no hay resultados
    const noResultsMessage = container.querySelector('.no-results');
    if (noResultsMessage) {
        noResultsMessage.style.display = visibleCount === 0 ? 'block' : 'none';
    }
}

// Función para formatear números de teléfono
function formatPhoneNumber(input) {
    let value = input.value.replace(/\D/g, '');
    if (value.length >= 10) {
        value = value.substring(0, 10);
        value = value.replace(/(\d{3})(\d{3})(\d{4})/, '($1) $2-$3');
    }
    input.value = value;
}

// Función para mostrar notificaciones toast
function showToast(message, type = 'info') {
    const toastContainer = document.getElementById('toast-container') || createToastContainer();
    
    const toast = document.createElement('div');
    toast.className = `toast align-items-center text-white bg-${type} border-0`;
    toast.setAttribute('role', 'alert');
    toast.innerHTML = `
        <div class="d-flex">
            <div class="toast-body">
                ${message}
            </div>
            <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
        </div>
    `;

    toastContainer.appendChild(toast);
    
    const bsToast = new bootstrap.Toast(toast);
    bsToast.show();

    // Remover el toast después de que se oculte
    toast.addEventListener('hidden.bs.toast', () => {
        toast.remove();
    });
}

function createToastContainer() {
    const container = document.createElement('div');
    container.id = 'toast-container';
    container.className = 'toast-container position-fixed top-0 end-0 p-3';
    container.style.zIndex = '1055';
    document.body.appendChild(container);
    return container;
}

// Función para confirmar acciones
function confirmAction(message, callback) {
    if (confirm(message)) {
        callback();
    }
}

// Función para hacer peticiones AJAX
async function makeAjaxRequest(url, method = 'GET', data = null) {
    try {
        const options = {
            method: method,
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            }
        };

        if (data) {
            options.body = JSON.stringify(data);
        }

        const response = await fetch(url, options);
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }

        return await response.json();
    } catch (error) {
        console.error('Error en petición AJAX:', error);
        showToast('Error en la comunicación con el servidor', 'danger');
        throw error;
    }
}

// Función para validar fechas
function validateDates() {
    const checkIn = document.getElementById('CheckInDate');
    const checkOut = document.getElementById('CheckOutDate');
    
    if (checkIn && checkOut) {
        const today = new Date().toISOString().split('T')[0];
        checkIn.min = today;
        
        checkIn.addEventListener('change', function() {
            const selectedDate = new Date(this.value);
            const minCheckOut = new Date(selectedDate);
            minCheckOut.setDate(minCheckOut.getDate() + 1);
            checkOut.min = minCheckOut.toISOString().split('T')[0];
            
            if (checkOut.value && new Date(checkOut.value) <= selectedDate) {
                checkOut.value = '';
            }
            
            calculatePrice();
        });

        checkOut.addEventListener('change', calculatePrice);
    }
}

// Función para inicializar componentes cuando el DOM esté listo
document.addEventListener('DOMContentLoaded', function() {
    // Inicializar validación de fechas
    validateDates();

    // Inicializar cálculo de precios
    const guestsInput = document.getElementById('NumberOfGuests');
    if (guestsInput) {
        guestsInput.addEventListener('change', calculatePrice);
    }

    // Inicializar tooltips de Bootstrap
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });

    // Inicializar popovers de Bootstrap
    const popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'));
    popoverTriggerList.map(function (popoverTriggerEl) {
        return new bootstrap.Popover(popoverTriggerEl);
    });

    // Inicializar dropdowns de Bootstrap
    const dropdownElementList = [].slice.call(document.querySelectorAll('.dropdown-toggle'));
    dropdownElementList.map(function (dropdownToggleEl) {
        return new bootstrap.Dropdown(dropdownToggleEl);
    });

    // Smooth scrolling para enlaces internos
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            if (target) {
                target.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });

    // Auto-hide alerts después de 5 segundos
    const alerts = document.querySelectorAll('.alert:not(.alert-permanent)');
    alerts.forEach(alert => {
        setTimeout(() => {
            const bsAlert = new bootstrap.Alert(alert);
            bsAlert.close();
        }, 5000);
    });

    // Formatear campos de teléfono
    const phoneInputs = document.querySelectorAll('input[type="tel"]');
    phoneInputs.forEach(input => {
        input.addEventListener('input', () => formatPhoneNumber(input));
    });
});

// Función para exportar datos (simulada)
function exportData(format = 'excel') {
    showToast(`Exportando datos en formato ${format.toUpperCase()}...`, 'info');
    
    // Simular descarga
    setTimeout(() => {
        showToast(`Datos exportados exitosamente en formato ${format.toUpperCase()}`, 'success');
    }, 2000);
}

// Función para actualizar estado de reserva via AJAX
async function updateReservationStatus(reservationId, newStatus) {
    try {
        const response = await makeAjaxRequest('/Admin/UpdateReservationStatus', 'POST', {
            reservationId: reservationId,
            status: newStatus
        });

        if (response.success) {
            showToast('Estado de reserva actualizado exitosamente', 'success');
            
            // Actualizar la UI
            const statusBadge = document.querySelector(`[data-reservation-id="${reservationId}"] .status-badge`);
            if (statusBadge) {
                statusBadge.textContent = newStatus;
                statusBadge.className = `badge status-badge bg-${getStatusColor(newStatus)}`;
            }
        } else {
            showToast(response.message || 'Error al actualizar el estado', 'danger');
        }
    } catch (error) {
        showToast('Error al actualizar el estado de la reserva', 'danger');
    }
}

// Función auxiliar para obtener color del estado
function getStatusColor(status) {
    switch (status.toLowerCase()) {
        case 'confirmada': return 'success';
        case 'pendiente': return 'warning';
        case 'cancelada': return 'danger';
        case 'completada': return 'info';
        default: return 'secondary';
    }
}

// Exportar funciones para uso global
window.GlampingParadise = {
    togglePassword,
    calculatePrice,
    filterItems,
    formatPhoneNumber,
    showToast,
    confirmAction,
    makeAjaxRequest,
    validateDates,
    exportData,
    updateReservationStatus
};